
import os
import re
import time
import random
from pathlib import Path


from qgis.PyQt import uic, QtWidgets
from qgis.PyQt.QtWidgets import QApplication, QDialog, QMessageBox, QWidget


from qgis.core import QgsVectorLayer


import numpy as np
import pandas as pd
import geopandas as gpd
from shapely.geometry.base import BaseGeometry


import common
from urbanq.logging.logging_config import logger
from urbanq.function.qss import gradient_style, default_style


from urbanq.function.file import (
    export_gdf,
    keep_columns_gdf,
    load_geojson_gdf,
    load_txt_or_csv_df,
    load_json_df_or_gdf,
    load_layer_or_shp_gdf,
    update_shapefile_layer,
    df_to_empty_geometry_gdf,
)

from urbanq.function.geo import (
    is_real_null,
    is_empty_value,
    normalize_null_values,
)

from urbanq.function.widgetutils import (
    show_progress,
    update_progress,
)


from urbanq.menu.autoUI.fileRread_dockwidget import fileRreadDockWidget
from urbanq.menu.autoUI.fileSave_dockwidget import fileSaveDockWidget
from urbanq.menu.autoUI.fileSetting_dockwidget import fileSettingDockWidget
from urbanq.menu.autoUI.ImageDescription_dockwidget import ImageDescriptionDockWidget



FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'DataCleaning_dockwidget_base.ui'))


class DataCleaningDockWidget(QDialog, FORM_CLASS):  
    def __init__(self, parent=None):
        
        super(DataCleaningDockWidget, self).__init__(parent)  
        
        
        
        
        
        self.setupUi(self)

        
        show_progress(self.progressBar, False)

        
        self.menuPushButton.setProperty("class", "boldText")
        self.nextStepPushButton.setProperty("class", "boldText")
        self.previousStepPushButton.setProperty("class", "boldText")

        
        self.menuPushButton.clicked.connect(self.go_back_to_data_conversion)

        
        self.nextStepPushButton.clicked.connect(lambda: self.next_previous_clicked(1))
        self.nextStepPushButton.clicked.connect(lambda: self.update_current_progress(self.stackedWidget.currentIndex()))
        self.nextStepPushButton.clicked.connect(lambda: self.load_menu_ui(self.stackedWidget.currentIndex()))

        
        self.previousStepPushButton.clicked.connect(lambda: self.next_previous_clicked(-1))
        self.previousStepPushButton.clicked.connect(lambda: self.update_current_progress(self.stackedWidget.currentIndex()))
        self.previousStepPushButton.clicked.connect(lambda: self.load_menu_ui(self.stackedWidget.currentIndex()))

        
        self.job_index = common.job_info.get("job_index") if common.job_info else None
        self.job_title = common.job_info.get("job_title") if common.job_info else None

        
        self.option = self.get_widget_option(self.job_index, self.job_title)

        
        self.pages_and_files = self.configure_pages_and_files()

        
        self.update_current_progress(0)

        
        self.stackedWidget.setCurrentIndex(0)

        
        self.load_menu_ui(0)

    
    
    

    def configure_pages_and_files(self):
        
        try:
            pages = []

            
            pages.append((True, self.current_step_1, ImageDescriptionDockWidget, None, None))

            
            pages.append((True, self.current_step_2, fileRreadDockWidget, self.option, None))

            
            read_required = any([
                self.option["setting_by_text"],
                self.option["setting_by_array"],
                self.option["setting_by_expression"],
                self.option["setting_by_section"]["enabled"],
                self.option["setting_by_numeric"]["enabled"],
                self.option["setting_by_combo"]["enabled"],
            ])
            pages.append((read_required, self.current_step_3, fileSettingDockWidget, self.option, None))

            
            save_required = any([
                self.option["output_by_file"],
                self.option["output_by_field"],
                self.option["output_by_table"]
            ])
            pages.append((save_required, self.current_step_4, fileSaveDockWidget, self.option, None))

            return pages

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def go_back_to_data_conversion(self):
        
        try:
            from urbanq.menu.dataConversion.dataConversion_dockwidget import dataConversionDockWidget  
            parent_ui = dataConversionDockWidget(self)  
            main_page_layout = self.parent().parent().findChild(QWidget, "page_dataConversion").layout()
            if main_page_layout:
                
                for i in reversed(range(main_page_layout.count())):
                    main_page_layout.itemAt(i).widget().deleteLater()
                main_page_layout.addWidget(parent_ui)

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def load_menu_ui(self, index):
        
        try:
            widget_enabled, widget_process, widget_class, widget_option, widget_instance = self.pages_and_files[index]
            page = self.stackedWidget.widget(index)

            
            if widget_instance is None:

                
                widget_instance = widget_class(self, self.option)
                page.layout().addWidget(widget_instance)
                self.pages_and_files[index] = (
                    self.pages_and_files[index][0],
                    self.pages_and_files[index][1],
                    self.pages_and_files[index][2],
                    self.pages_and_files[index][3],
                    widget_instance
                )

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def update_current_progress(self, index):
        
        try:
            step = 1
            for i, (widget_enabled, widget_process, _, _, _) in enumerate(self.pages_and_files):
                if not widget_enabled:
                    widget_process.hide()
                    continue
                else:
                    updated_text = re.sub(r"\[\d+단계\]", f"[{step}단계]", widget_process.text())
                    widget_process.setText(updated_text)
                    step += 1

                
                widget_process.show()

                if i == index:
                    widget_process.setStyleSheet(gradient_style)
                else:
                    widget_process.setStyleSheet(default_style)

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def get_safe_page_index(self, current_index: int, direction: int) -> int:
        
        try:
            new_index = current_index

            while True:
                
                new_index += direction

                
                new_index = max(0, min(new_index, len(self.pages_and_files) - 1))

                
                if self.pages_and_files[new_index][0]:
                    return new_index

                
                if new_index == 0 and direction == -1:
                    return current_index

                
                if new_index == len(self.pages_and_files) - 1 and direction == 1:
                    return current_index

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def next_previous_clicked(self, direction):
        
        def get_last_valid_page_index(pages_and_files) -> int:
            
            for i in reversed(range(len(pages_and_files))):
                if pages_and_files[i][0]:
                    return i
            return -1  

        try:
            
            current_index = self.stackedWidget.currentIndex()

            
            if self.pages_and_files[current_index][0]:
                instance = self.pages_and_files[current_index][4]
                if direction > 0 and not instance.set_fileResults():
                    return

            
            new_index = self.get_safe_page_index(current_index, direction)

            
            last_page_index = get_last_valid_page_index(self.pages_and_files)

            
            self.nextStepPushButton.setText("실행하기 " if new_index == last_page_index else "다음 단계 ▶")

            
            self.stackedWidget.setCurrentIndex(new_index)

            
            if current_index == last_page_index and direction > 0:
                self.run_job_process()

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    
    
    

    def get_file_data_frame(self, source_file_type, source_file_path, file_path, file_encoding, file_delimiter, file_has_header):
        
        try:
            
            gdf = None

            
            if source_file_type == "shp":
                gdf = load_layer_or_shp_gdf(shp_path=file_path, file_encoding=file_encoding)

            
            elif source_file_type == "layer":
                qgs_project_layer = source_file_path
                gdf = load_layer_or_shp_gdf(layer=qgs_project_layer, file_encoding=file_encoding)

            
            elif source_file_type == "json":
                df, _ = load_json_df_or_gdf(file_path=file_path, file_encoding=file_encoding)
                gdf = df_to_empty_geometry_gdf(df)

            
            elif source_file_type == "geojson":
                gdf = load_geojson_gdf(file_path=file_path, file_encoding=file_encoding)

            
            elif source_file_type == "txt":
                df = load_txt_or_csv_df(file_path, file_encoding, file_delimiter, file_has_header)
                gdf = df_to_empty_geometry_gdf(df)

            
            elif source_file_type == "csv":
                df = load_txt_or_csv_df(file_path, file_encoding, file_delimiter, file_has_header)
                gdf = df_to_empty_geometry_gdf(df)

            
            elif source_file_type == "folder":
                df = load_txt_or_csv_df(file_path, file_encoding, file_delimiter, file_has_header)
                gdf = df_to_empty_geometry_gdf(df)

            if gdf is None:
                return

            return gdf

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def run_job_process(self):
        
        try:
            
            show_progress(self.progressBar)

            
            total_files = len(common.fileInfo_1.file_preview)  
            steps_per_file = 4  
            total_steps = total_files * steps_per_file  
            base_progress = 20  
            step_weight = (100 - base_progress) / total_steps  
            current_step = 0  

            
            source_file_type, source_file_path, _ = common.fileInfo_1.file_record.get_record()
            result_file_type, result_file_path, _ = common.fileInfo_1.result_record.get_record()

            
            status_flags = []  
            for index, file_preview in enumerate(common.fileInfo_1.file_preview):

                
                file_path, file_encoding, file_delimiter, file_has_header = file_preview.get_info()
                current_step += 1
                update_progress(self.progressBar, int(base_progress + current_step * step_weight))

                
                if source_file_type == "folder":
                    
                    file_name_with_ext = os.path.basename(file_path)
                    new_file_path = os.path.join(result_file_path, file_name_with_ext)
                elif result_file_type == "layer":
                    new_file_path = file_path
                else:
                    new_file_path = result_file_path

                
                gdf = self.get_file_data_frame(source_file_type, source_file_path, file_path, file_encoding, file_delimiter, file_has_header)
                current_step += 1
                update_progress(self.progressBar, int(base_progress + current_step * step_weight))

                
                result = self.run_job_by_index(gdf, index)
                current_step += 1
                update_progress(self.progressBar, int(base_progress + current_step * step_weight))

                
                if result is None:
                    status_flags.append(False)
                    break
                elif result is True:
                    
                    
                    status_flags.append(True)

                try:
                    
                    if result_file_type == 'layer':

                        
                        layer_widget = self.pages_and_files[1][4].get_qgs_layer_widget()

                        
                        layer_widget_index = layer_widget.currentIndex()

                        
                        layer = source_file_path

                        
                        new_layer = update_shapefile_layer(layer, result)

                        
                        if 0 <= layer_widget_index < layer_widget.count():
                            layer_widget.setCurrentIndex(layer_widget_index)

                        
                        common.fileInfo_1.file_record.file_path[result_file_type] = new_layer

                        
                        status_flags.append(True)

                    else:
                        
                        if new_file_path:

                            
                            if isinstance(result, gpd.GeoDataFrame):
                                export_success = export_gdf(result, new_file_path)

                                
                                status_flags.append(export_success)

                            elif isinstance(result, list) and result:
                                
                                file_type, _, file_name = common.fileInfo_1.file_record.get_record()
                                base_dir = Path(new_file_path)
                                base_name = Path(file_name).stem
                                ext = f".{file_type}"

                                
                                export_success = []
                                for i, part in enumerate(result, start=1):
                                    output_path = base_dir / f"{base_name}_{i:03d}{ext}"
                                    export_success.append(export_gdf(part, output_path))

                                
                                status_flags.append(all(export_success))

                            else:
                                
                                QMessageBox.information(self, "파일 오류", "파일 저장 중 오류가 발생했습니다.", QMessageBox.Ok)
                                status_flags.append(False)

                except Exception as e:
                    
                    QMessageBox.information(self, "파일 오류", f"GeoDataFrame export 실패: {e}", QMessageBox.Ok)
                    status_flags.append(False)

                
                current_step += 1
                update_progress(self.progressBar, int(base_progress + current_step * step_weight))

            
            if status_flags and all(status_flags):
                update_progress(self.progressBar, 100)  
                QMessageBox.information(self, "알림", "축하합니다. 작업이 완료했습니다!", QMessageBox.Ok)

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

        finally:
            show_progress(self.progressBar, False)

    
    
    

    
    def calculate_value_frequency_statistics(self, gdf, target_field):
        
        try:
            
            geometry_col = gdf.geometry.name

            
            gdf_copy = gdf.copy()

            
            gdf_copy_no_geom = gdf_copy.drop(columns=geometry_col, errors='ignore')

            
            col_series = normalize_null_values(gdf_copy_no_geom[[target_field]])

            
            value_counts = col_series.value_counts(dropna=False)

            
            header = ["필드 이름", "값", "빈도수"]
            rows = []
            for value, count in value_counts.items():
                value_display = str(value) if pd.notnull(value) else "NULL"
                rows.append([target_field, value_display, int(count)])

            
            common.fileInfo_1.result_table["header"] = header
            common.fileInfo_1.result_table["rows"] = rows
            common.fileInfo_1.result_table["msg"] = (
                f"'{target_field}' 필드의 고유 값 개수: {len(rows)}"
            )

            
            common.signals.file_preview_updated.emit()

            return True

        except Exception as e:
            QMessageBox.information(self, "작업 오류", "속성 값 빈도 계산 중 오류가 발생하였습니다.", QMessageBox.Ok)
            logger.error("에러 발생: %s", e, exc_info=True)
            return False

    
    def initialize_fields_by_type(self, gdf, target_fields, array_string="N/A", array_integer=0, array_float=0.0):
        
        def is_int_or_float(series: pd.Series) -> str:
            
            if pd.api.types.is_integer_dtype(series):
                return "int"
            elif pd.api.types.is_float_dtype(series):
                return "float"

            
            
            try:
                numeric_series = pd.to_numeric(series, errors='raise')

                
                s_no_na = numeric_series.dropna()
                if s_no_na.empty:
                    return "int"  

                return "int" if (s_no_na % 1 == 0).all() else "float"
            except:
                return "string"

        def initialize_values(gdf, fields):
            for col in fields:
                col_type = is_int_or_float(gdf[col])

                if col_type == "int":
                    gdf[col] = array_integer
                elif col_type == "float":
                    gdf[col] = array_float
                else:
                    gdf[col] = array_string
            return gdf

        try:
            
            geometry_col = gdf.geometry.name

            
            gdf_copy = gdf.copy()

            
            gdf_copy_no_geom = gdf_copy.drop(columns=geometry_col, errors='ignore')

            
            gdf_copy_not_null_values = normalize_null_values(gdf_copy_no_geom)

            
            gdf_copy_not_null_values = initialize_values(gdf_copy_not_null_values, target_fields)

            
            gdf_copy.update(gdf_copy_not_null_values)

            
            return gdf_copy

        except Exception as e:
            QMessageBox.information(self, "작업 오류", "필드 초기화 중 오류가 발생하였습니다.", QMessageBox.Ok)
            logger.error("에러 발생: %s", e, exc_info=True)
            return None

    
    def combine_fields_with_delimiter(self, gdf, field_list, delimiter, result_field):
        
        try:
            
            gdf_copy = gdf.copy()

            
            gdf_copy = normalize_null_values(gdf_copy)

            
            gdf_copy[result_field] = gdf_copy[field_list].astype(str).apply(
                lambda row: delimiter.join(row.values), axis=1
            )

            return gdf_copy

        except Exception as e:
            QMessageBox.information(self, "작업 오류", "필드 결합 중 오류가 발생하였습니다.", QMessageBox.Ok)
            logger.error("에러 발생: %s", e, exc_info=True)
            return None

    
    def split_date_field_to_ymd(self, gdf, target_field):
        
        try:
            
            geometry_col = gdf.geometry.name

            
            gdf_copy = gdf.copy()

            
            gdf_copy_no_geom = gdf_copy.drop(columns=geometry_col, errors='ignore')

            
            gdf_copy_not_null_values = normalize_null_values(gdf_copy_no_geom)

            
            s = gdf_copy_not_null_values[target_field]

            
            date_series = pd.to_datetime(s, errors='coerce')

            
            
            candidate_formats = [
                "%Y-%m-%d",
                "%Y/%m/%d",
                "%Y.%m.%d",
                "%Y%m%d",
                "%Y-%m-%d %H:%M:%S",
                "%Y/%m/%d %H:%M:%S",
                "%Y.%m.%d %H:%M:%S",
                "%Y-%m-%d %H:%M",
                "%Y/%m/%d %H:%M",
                "%Y.%m.%d %H:%M",
                "%m/%d/%Y",
            ]

            remaining_mask = date_series.isna() & s.notna()
            if remaining_mask.any():
                for fmt in candidate_formats:
                    
                    parsed = pd.to_datetime(s[remaining_mask], format=fmt, errors='coerce')
                    
                    filled_mask = parsed.notna()
                    if filled_mask.any():
                        date_series.loc[parsed.index[filled_mask]] = parsed[filled_mask]
                        remaining_mask = date_series.isna() & s.notna()
                        if not remaining_mask.any():
                            break

            
            if date_series.isna().all():
                QMessageBox.information(self, "변환 실패", f"'{target_field}' 필드는 날짜 형식으로 변환할 수 없습니다.", QMessageBox.Ok)
                return None

            
            def get_unique_field(base_name):
                name = base_name
                count = 0
                while name in gdf_copy.columns:
                    count += 1
                    name = base_name + ("_" * count)
                return name

            
            year_field = get_unique_field("Year")
            month_field = get_unique_field("Month")
            day_field = get_unique_field("Day")

            
            
            gdf_copy[year_field] = date_series.dt.year.astype("Int64")
            gdf_copy[month_field] = date_series.dt.month.astype("Int64")
            gdf_copy[day_field] = date_series.dt.day.astype("Int64")

            return gdf_copy

        except Exception as e:
            QMessageBox.information(self, "작업 오류", "날짜 필드 분리 중 오류가 발생하였습니다.", QMessageBox.Ok)
            logger.error("에러 발생: %s", e, exc_info=True)
            return None

    
    
    

    @staticmethod
    def get_widget_option(job_index, job_title):
        
        try:
            option = None  
            job_title = job_title[2:]

            if job_index == 0:
                option = {
                    "apply_basic_qss": True,

                    "disable_file_type_layer": True,
                    "disable_file_type_shp": True,
                    "disable_file_type_json": True,
                    "disable_file_type_txtcsv": True,
                    "disable_file_type_fold": False,

                    "show_uid_in_file": False,
                    "show_tuid_in_file": True,
                    "show_field_in_file": False,

                    "setting_by_text": False,
                    "setting_by_array": False,
                    "setting_by_expression": False,
                    "setting_by_section": {"enabled": False, "value_type": "int"},
                    "setting_by_numeric": {"enabled": False, "value_type": "int"},
                    "setting_by_combo": {"enabled": False, "items": []},

                    "output_by_file": False,
                    "output_by_field": False,
                    "output_by_table": True,

                    "FILE_TUID": [
                        '빈도 계산 기준 필드 선택',
                        '필드 선택: ',
                        "각 속성 값의 출현 빈도를 계산하여 통계합니다."
                    ],

                    "RESULT_TABLE": [
                        f'{job_title} 결과 테이블',
                        ''
                    ]
                }
            if job_index == 1:
                option = {
                    "apply_basic_qss": True,

                    "disable_file_type_layer": True,
                    "disable_file_type_shp": True,
                    "disable_file_type_json": True,
                    "disable_file_type_txtcsv": True,
                    "disable_file_type_fold": True,

                    "show_uid_in_file": False,
                    "show_tuid_in_file": False,
                    "show_field_in_file": True,

                    "setting_by_text": False,
                    "setting_by_array": True,
                    "setting_by_expression": False,
                    "setting_by_section": {"enabled": False, "value_type": "int"},
                    "setting_by_numeric": {"enabled": False, "value_type": "int"},
                    "setting_by_combo": {"enabled": False, "items": []},

                    "output_by_file": True,
                    "output_by_field": False,
                    "output_by_table": False,

                    "FILE_FIELD": [
                        '일괄 수정할 필드 선택 (복수 선택 가능)',
                        '필드 선택: ',
                        '선택한 필드의 유형을 정수형(Integer), 실수형(Float), 문자열(String)로 자동 감지합니다.\n'
                        '감지된 필드 유형에 따라 다음 단계에서 각 유형별 수정 값을 개별적으로 설정할 수 있습니다.'
                    ],

                    "SETTING_ARRAY": [
                        '필드 유형별 수정 값 설정',
                        '문자열(String) 필드 수정 값: ',
                        '정수(Integer) 필드 수정 값: ',
                        '실수(Float) 필드 수정 값: ',
                        '각 필드 유형에 대해 적용할 값을 입력해 주세요.'
                    ],
                }
            if job_index == 2:
                option = {
                    "apply_basic_qss": True,

                    "disable_file_type_layer": True,
                    "disable_file_type_shp": True,
                    "disable_file_type_json": True,
                    "disable_file_type_txtcsv": True,
                    "disable_file_type_fold": False,

                    "show_uid_in_file": False,
                    "show_tuid_in_file": False,
                    "show_field_in_file": True,

                    "setting_by_text": True,
                    "setting_by_array": False,
                    "setting_by_expression": False,
                    "setting_by_section": {"enabled": False, "value_type": "int"},
                    "setting_by_numeric": {"enabled": False, "value_type": "int"},
                    "setting_by_combo": {"enabled": False, "items": []},

                    "output_by_file": True,
                    "output_by_field": True,
                    "output_by_table": False,

                    "FILE_FIELD": ['결합할 필드 선택 (복수 선택 가능)',
                                   '필드 선택: ',
                                   '선택한 여러 필드의 값을 지정한 순서대로 특정 부호 또는 구분자로 연결하여 하나의 필드로 결합합니다.'],

                    "SETTING_TEXT": ['필드 결합에 사용할 부호 또는 구분자 설정',
                                     '부호 또는 구분자 입력: ',
                                     '선택한 필드를 부호 또는 구분자(예: -, /, . 등)로 연결하여 하나의 필드로 결합합니다.'],

                    "RESULT_FIELD": ['결합 결과 필드 생성', '필드명 입력: ', ''],
                }
            if job_index == 3:
                option = {
                    "apply_basic_qss": True,

                    "disable_file_type_layer": True,
                    "disable_file_type_shp": True,
                    "disable_file_type_json": True,
                    "disable_file_type_txtcsv": True,
                    "disable_file_type_fold": False,

                    "show_uid_in_file": False,
                    "show_tuid_in_file": True,
                    "show_field_in_file": False,

                    "setting_by_text": False,
                    "setting_by_array": False,
                    "setting_by_expression": False,
                    "setting_by_section": {"enabled": False, "value_type": "int"},
                    "setting_by_numeric": {"enabled": False, "value_type": "int"},
                    "setting_by_combo": {"enabled": False, "items": []},

                    "output_by_file": True,
                    "output_by_field": False,
                    "output_by_table": False,

                    "FILE_TUID": ['날짜 필드 선택',
                                  '필드 선택: ',
                                  "선택한 필드의 날짜 값을 자동으로 인식하여 연도(Year), 월(Month), 일(Day) 필드로 분리합니다.\n"
                                  "예: 2024-05-25, 2024.05.25, 2024/05/25, 20240525, 24-05-25, 05/25/2024"]
                }
            return option

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def run_job_by_index(self, gdf, file_preview_index):
        
        try:
            
            file_info = common.fileInfo_1

            
            setting_text = file_info.file_setting.get_text()
            setting_numeric = file_info.file_setting.get_numeric()
            setting_section_min, setting_section_max = file_info.file_setting.get_section()
            setting_combo = file_info.file_setting.get_combo()
            setting_array_string, setting_array_integer, setting_array_float = file_info.file_setting.get_array()

            
            source_file_type, source_file_path, source_file_name = file_info.file_record.get_record()

            
            file_preview = file_info.file_preview[file_preview_index]
            file_fieldes = file_preview.get_header()
            file_field_selection = file_preview.get_selection_field()
            file_tuid = file_preview.get_file_tuid()
            file_is_field_check = file_preview.get_field_check()
            result_field = file_info.result_field

            
            gdf.columns = gdf.columns.astype(str)

            
            result = None
            if self.job_index == 0:
                result = self.calculate_value_frequency_statistics(gdf, file_tuid)

            elif self.job_index == 1:
                if not file_is_field_check:
                    file_field_selection = file_fieldes
                result = self.initialize_fields_by_type(gdf, file_field_selection, setting_array_string, setting_array_integer, setting_array_float)

            elif self.job_index == 2:
                if not file_is_field_check:
                    file_field_selection = file_fieldes
                result = self.combine_fields_with_delimiter(gdf, file_field_selection, setting_text, result_field)

            elif self.job_index == 3:
                result = self.split_date_field_to_ymd(gdf, file_tuid)

            
            if result is None or result is False:
                return None

            return result

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)
            return None




